<?php

namespace App\Http\Controllers;

use App\Models\Pelicula;
use App\Models\Categoria;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf; // Importa la clase PDF

class PeliculaController extends Controller
{
    public function index()
    {
        // Obtener las películas paginadas (10 películas por página)
        $peliculas = Pelicula::with('categoria')->paginate(10);

        return view('peliculas.index', compact('peliculas'));
    }

    public function create()
    {
        $categorias = Categoria::all();
        return view('peliculas.create', compact('categorias'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'titulo' => 'required|string|max:255',
            'duracion' => 'required|integer',
            'formato' => 'required|string|max:255',
            'categoria_id' => 'required|exists:categorias,id',
            'estado' => 'required|in:disponible,alquilada',
        ]);

        Pelicula::create($request->all());

        return redirect()->route('peliculas.index')->with('success', 'Película creada correctamente.');
    }

    public function show(Pelicula $pelicula)
    {
        return view('peliculas.show', compact('pelicula'));
    }

    public function edit(Pelicula $pelicula)
    {
        $categorias = Categoria::all();
        return view('peliculas.edit', compact('pelicula', 'categorias'));
    }

    public function update(Request $request, Pelicula $pelicula)
    {
        $request->validate([
            'titulo' => 'required|string|max:255',
            'duracion' => 'required|integer',
            'formato' => 'required|string|max:255',
            'categoria_id' => 'required|exists:categorias,id',
            'estado' => 'required|in:disponible,alquilada',
        ]);

        $pelicula->update($request->all());

        return redirect()->route('peliculas.index')->with('success', 'Película actualizada correctamente.');
    }

    public function destroy(Pelicula $pelicula)
    {
        $pelicula->delete();
        return redirect()->route('peliculas.index')->with('success', 'Película eliminada correctamente.');
    }

    public function search(Request $request)
    {
        // Obtener el término de búsqueda desde la solicitud
        $term = $request->get('term');

        // Realizar la búsqueda utilizando el término
        if ($term) {
            $peliculas = Pelicula::where('titulo', 'like', '%' . $term . '%')->get();
        } else {
            $peliculas = [];
        }

        // Retornar las películas encontradas como respuesta JSON
        return response()->json([
            'peliculas' => $peliculas
        ]);
    }

    // Método para exportar las películas a PDF
    public function exportToPDF()
    {
        // Obtener todas las películas con su categoría
        $peliculas = Pelicula::with('categoria')->get();

        // Cargar la vista y pasar los datos
        $pdf = Pdf::loadView('peliculas.pdf', compact('peliculas'));

        // Descargar el archivo PDF
        return $pdf->download('peliculas.pdf');
    }
}
