<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Cobro;
use App\Models\Alquiler;

class CobroController extends Controller
{
    /**
     * Muestra la lista de cobros con paginación, incluyendo el precio de la película y el vuelto.
     */
    public function index()
    {
        // Obtener todos los cobros con su relación alquiler
        $cobros = Cobro::with('alquiler.pelicula')->latest()->paginate(10);
    
        $totalesPorPelicula = Alquiler::join('peliculas', 'alquileres.pelicula_id', '=', 'peliculas.id')
    ->selectRaw('peliculas.titulo AS nombre_pelicula, SUM(alquileres.precio) AS total_precio')
    ->groupBy('peliculas.titulo')
    ->pluck('total_precio', 'nombre_pelicula');

    
        return view('cobros.index', compact('cobros', 'totalesPorPelicula'));
    }
    

    /**
     * Guarda un nuevo cobro en la base de datos, calculando el vuelto.
     */
    public function store(Request $request)
    {
        $request->validate([
            'alquiler_id' => 'required|exists:alquileres,id',
            'monto' => 'required|numeric|min:0',
        ]);

        // Buscar el alquiler correspondiente
        $alquiler = Alquiler::findOrFail($request->alquiler_id);

        // Obtener el precio de la película
        $precioPelicula = $alquiler->precio;

        // Calcular el vuelto
        $vuelto = $request->monto - $precioPelicula;

        // Registrar el cobro
        Cobro::create([
            'alquiler_id' => $alquiler->id,
            'monto' => $request->monto,
            'vuelto' => $vuelto,
            'fecha_cobro' => now(),
        ]);

        // Actualizar el estado del alquiler a "pagado"
        $alquiler->estado = 'pagado';
        $alquiler->save();

        return back()->with('success', 'Cobro registrado correctamente.');
    }
}
