<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Alquiler;
use App\Models\Pelicula;
use App\Models\Sala;
use App\Models\Cliente;
use App\Models\Cobro;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;
use Barryvdh\DomPDF\Facade\Pdf;

class AlquilerController extends Controller
{
    public function index()
    {
        $alquileres = Alquiler::with(['pelicula', 'sala', 'cliente'])->paginate(10);
        return view('alquileres.index', compact('alquileres'));
    }

    public function create()
    {
        $peliculas = Pelicula::where('estado', 'disponible')->get();
        $salas = Sala::all();
        $clientes = Cliente::all();

        return view('alquileres.create', compact('peliculas', 'salas', 'clientes'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'pelicula_id' => 'required|exists:peliculas,id',
            'sala_id' => 'required|exists:salas,id',
            'cliente' => 'required|string|max:255',
            'hora_entrada' => 'required|date_format:H:i',
            'hora_salida' => 'required|date_format:H:i|after:hora_entrada',
            'precio' => 'required|numeric|min:0',
        ]);

        $pelicula = Pelicula::find($request->pelicula_id);

        if ($pelicula->estado !== 'disponible') {
            return redirect()->route('alquileres.create')
                ->withErrors(['error' => 'Esta película no está disponible para alquiler.']);
        }

        Alquiler::create($request->only(['pelicula_id', 'sala_id', 'cliente', 'hora_entrada', 'hora_salida', 'precio']));

        $pelicula->estado = 'alquilada';
        $pelicula->save();

        return redirect()->route('alquileres.index')->with('success', 'Alquiler registrado correctamente.');
    }

    public function show($id)
    {
        $alquiler = Alquiler::findOrFail($id);
        return view('alquileres.show', compact('alquiler'));
    }

    public function edit($id)
    {
        $alquiler = Alquiler::findOrFail($id);
        $peliculas = Pelicula::where('estado', 'disponible')->orWhere('id', $alquiler->pelicula_id)->get();
        $salas = Sala::all();
        $clientes = Cliente::all();

        return view('alquileres.edit', compact('alquiler', 'peliculas', 'salas', 'clientes'));
    }

    public function update(Request $request, $id)
    {
        $alquiler = Alquiler::findOrFail($id);

        $request->validate([
            'pelicula_id' => 'required|exists:peliculas,id',
            'cliente' => 'required|string|max:255',
            'hora_entrada' => 'required|date_format:H:i',
            'hora_salida' => 'required|date_format:H:i|after:hora_entrada',
            'precio' => 'required|numeric|min:0',
        ]);

        Log::info('Datos recibidos para actualización:', $request->all());

        $data = $request->only(['pelicula_id', 'cliente', 'hora_entrada', 'hora_salida', 'precio']);

        $alquiler->update($data);

        return redirect()->route('alquileres.index')->with('success', 'Alquiler actualizado correctamente.');
    }

    public function destroy($id)
    {
        try {
            $alquiler = Alquiler::findOrFail($id);

            if ($alquiler->pelicula) {
                Pelicula::where('id', $alquiler->pelicula_id)->update(['estado' => 'disponible']);
            }

            $alquiler->delete();

            return redirect()->route('alquileres.index')->with('success', 'Alquiler eliminado correctamente.');
        } catch (ModelNotFoundException $e) {
            return redirect()->route('alquileres.index')->with('error', 'Alquiler no encontrado.');
        }
    }

    public function cobrar(Request $request, $id)
    {
        Log::info('Datos recibidos:', $request->all());

        $monto = $request->input('monto', null);

        if (is_null($monto) || $monto === '') {
            return response()->json(['mensaje' => 'Monto no recibido'], 400);
        }

        $alquiler = Alquiler::findOrFail($id);

        if ($monto < $alquiler->precio) {
            return response()->json(['mensaje' => 'Monto insuficiente'], 400);
        }

        // Registrar el cobro en la base de datos
        $cobro = Cobro::create([
            'alquiler_id' => $alquiler->id,
            'monto' => $monto,
            'fecha_cobro' => now(),
        ]);

        // Actualizar el estado del alquiler a "pagado"
        $alquiler->estado = 'pagado';
        $alquiler->save();

        return response()->json([
            'mensaje' => 'Pago realizado con éxito',
            'cobro' => $cobro,
        ]);
    }

    public function generarTicket($id)
    {
        $alquiler = Alquiler::findOrFail($id);

        $pdf = Pdf::loadView('alquileres.ticket', compact('alquiler'))
            ->setPaper([0, 0, 226.500, 600], 'portrait');

        return $pdf->stream("ticket_alquiler_{$alquiler->id}.pdf");
    }
}
